<?php
/* --------------------------------------------------------------
   ParcelServiceWriter.php 2020-04-16
   Gambio GmbH
   http://www.gambio.de
   Copyright (c) 2019 Gambio GmbH
   Released under the GNU General Public License (Version 2)
   [http://www.gnu.org/licenses/gpl-2.0.html]
   --------------------------------------------------------------
*/

declare(strict_types=1);

namespace Gambio\Admin\ParcelService\Repository;

use Doctrine\DBAL\Connection;
use Doctrine\DBAL\ConnectionException;
use Exception;
use Gambio\Admin\ParcelService\Interfaces\ParcelService;
use Gambio\Admin\ParcelService\Interfaces\ParcelServiceId;
use Gambio\Admin\ParcelService\Interfaces\ParcelServiceIds;
use Gambio\Admin\ParcelService\Interfaces\ParcelServices;
use Gambio\Core\Language\Language;
use Gambio\Core\Language\LanguageService;
use Gambio\Core\Language\Model\LanguageId;
use RuntimeException;

/**
 * Class ParcelServiceWriter
 *
 * @package Gambio\Admin\ParcelService\Repository
 */
class ParcelServiceWriter
{
    /**
     * @var Connection
     */
    private $db;
    /**
     * @var LanguageService
     */
    private $languageService;
    
    
    /**
     * ParcelServiceWriter constructor.
     *
     * @param Connection      $db
     * @param LanguageService $languageService
     */
    public function __construct(Connection $db, LanguageService $languageService)
    {
        $this->db              = $db;
        $this->languageService = $languageService;
    }
    
    
    /**
     * Inserts a new parcel service into the database.
     *
     * @param ParcelService|ParcelService $parcelService
     *
     * @return int
     */
    public function insert(ParcelService $parcelService): int
    {
        if ($parcelService->isDefault()) {
            $this->db->createQueryBuilder()->update('parcel_services')->set('`default`', '0')->execute();
        }
        
        $this->db->createQueryBuilder()
            ->insert('parcel_services')
            ->setValue('`name`', ':name')
            ->setValue('`default`', ':default')
            ->setParameter('name', $parcelService->name())
            ->setParameter('default', (int)$parcelService->isDefault())
            ->execute();
        $id = $this->db->createQueryBuilder()->getConnection()->lastInsertId();
        
        /** @var Language $language */
        foreach ($this->languageService->getAvailableLanguages() as $language) {
            $languageId = LanguageId::create($language->id());
            
            $this->db->createQueryBuilder()
                ->insert('parcel_services_description')
                ->setValue('parcel_service_id', ':parcel_service_id')
                ->setValue('language_id', ':language_id')
                ->setValue('url', ':url')
                ->setValue('comment', ':comment')
                ->setParameter('parcel_service_id', $id)
                ->setParameter('language_id', $language->id())
                ->setParameter('url', $parcelService->url($languageId))
                ->setParameter('comment', $parcelService->comment($languageId))
                ->execute();
        }
        
        return (int)$id;
    }
    
    
    /**
     * Inserts multiple parcel services into the database and returns the ID.
     *
     * @param ParcelServices $parcelServices
     *
     * @return int[]
     *
     * @throws ConnectionException
     *
     */
    public function insertMultiple(ParcelServices $parcelServices): array
    {
        $ids = [];
        try {
            $this->db->beginTransaction();
            foreach ($parcelServices as $reference => $parcelService) {
                
                $ids[$reference] = $this->insert($parcelService);
            }
            $this->db->commit();
        } catch (Exception $exception) {
            $this->db->rollBack();
            throw new RuntimeException('Could not add the parcel service', $exception->getCode(), $exception);
        }
        
        return $ids;
    }
    
    
    /**
     * Updates an existing parcel service in the database.
     *
     * @param ParcelService|ParcelService $parcelService
     */
    public function update(ParcelService $parcelService): void
    {
        if ($parcelService->isDefault()) {
            $this->db->createQueryBuilder()->update('parcel_services')->set('`default`', '0')->execute();
        }
        
        $this->db->createQueryBuilder()
            ->update('parcel_services')
            ->set('`name`', ':name')
            ->set('`default`', ':default')
            ->where('parcel_service_id = :id')
            ->setParameter('name', $parcelService->name())
            ->setParameter('default', (int)$parcelService->isDefault())
            ->setParameter('id', $parcelService->id())
            ->execute();
        
        /** @var Language $language */
        foreach ($this->languageService->getAvailableLanguages() as $language) {
            $languageId = LanguageId::create($language->id());
            
            $this->db->createQueryBuilder()
                ->update('parcel_services_description')
                ->set('url', ':url')
                ->set('comment', ':comment')
                ->where('parcel_service_id = :id')
                ->andwhere('language_id = :language_id')
                ->setParameter('url', $parcelService->url($languageId))
                ->setParameter('comment', $parcelService->comment($languageId))
                ->setParameter('id', $parcelService->id())
                ->setParameter('language_id', $language->id())
                ->execute();
        }
    }
    
    
    /**
     * Updates multiple existing parcel services in the database.
     *
     * @param ParcelServices $parcelServices
     *
     * @throws ConnectionException
     */
    public function updateMultiple(ParcelServices $parcelServices): void
    {
        try {
            $this->db->beginTransaction();
            foreach ($parcelServices as $parcelService) {
                $this->update($parcelService);
            }
            $this->db->commit();
        } catch (Exception $exception) {
            $this->db->rollBack();
            throw new RuntimeException('Could not update the parcel service', $exception->getCode(), $exception);
        }
    }
    
    
    /**
     * Deletes a parcel service from the database.
     *
     * @param ParcelServiceId $id
     */
    public function delete(ParcelServiceId $id): void
    {
        $this->db->createQueryBuilder()
            ->delete('parcel_services')
            ->where('parcel_service_id = :id')
            ->setParameter('id', $id->value())
            ->execute();
        
        $this->db->createQueryBuilder()
            ->delete('parcel_services_description')
            ->where('parcel_service_id = :id')
            ->setParameter('id', $id->value())
            ->execute();
    }
    
    
    /**
     * Deletes multiple parcel services from the database.
     *
     * @param ParcelServiceIds $ids
     *
     * @throws ConnectionException
     */
    public function deleteMultiple(ParcelServiceIds $ids): void
    {
        try {
            $this->db->beginTransaction();
            foreach ($ids as $id) {
                $this->delete($id);
            }
            $this->db->commit();
        } catch (Exception $exception) {
            $this->db->rollBack();
            throw new RuntimeException('Could not delete the parcel service', $exception->getCode(), $exception);
        }
    }
}